import os
from prompta.utils.set_api import apikey
from prompta.utils.config_helper import read_config, deep_merge_dicts
from prompta.core.env import BaseEnv
from prompta.core.env.control_primitives import load_control_primitives
from prompta.core.env.event_logger import EventLogger
from prompta.core.env.utils import UTILS
from voyager.env import VoyagerEnv


default_config = read_config(os.path.join(os.path.dirname(__file__), "default_mc_config.yaml"))

class MinecraftSimpleEnv(BaseEnv):
    def __init__(self, whole_cfg):
        super().__init__()
        self.whole_cfg = deep_merge_dicts(default_config, whole_cfg)
        self.config = self.whole_cfg.env
        self.setup_env()
        self.setup_scripts()
        self.setup_event_logger()

    def setup_env(self):
        existing_logs = set(os.listdir(os.path.join(self.whole_cfg.env.log_path, "mineflayer")))
        self.env = VoyagerEnv(
            mc_port=self.whole_cfg.env.mc_port,
            azure_login=self.whole_cfg.env.azure_login,
            server_port=self.whole_cfg.env.server_port,
            request_timeout=self.whole_cfg.env.request_timeout,
            log_path=self.whole_cfg.env.log_path,
        )
        current_logs = set(os.listdir(os.path.join(self.env.log_path, "mineflayer")))
        self.mf_log_path = os.path.join(self.whole_cfg.env.log_path, "mineflayer", (current_logs - existing_logs).pop())

    def setup_scripts(self):
        self.control_primitives = load_control_primitives()
        self.program_context = ""
        for primitives in self.control_primitives:
            self.program_context += f"{primitives}\n\n"

        for util in UTILS:
            self.program_context += f"{UTILS[util]}\n\n"

    def setup_event_logger(self):
        self.event_logger = EventLogger(self.config.event_log_path, self.mf_log_path)
        self.events_path = self.event_logger.event_log_path

    def reset(self):
        self.env.reset(
            options={
                "mode": "soft",
                "wait_ticks": self.whole_cfg.env.env_wait_ticks,
            }
        )
        self.env.unpause()
        events = self.env.step(
                "bot.chat(`/time set ${getNextTime()}`);\n"
                + f"bot.chat('/difficulty peaceful');"
            )
        self.clear_inventory()
        print("Clearing inventory...")
        self.tp_to_random_location()
        print("Teleporting to random location...")
        self.env.pause()
        self.chest_memory = {}
        self.event_logger.reset(events)
        self.events_path = self.event_logger.event_log_path
        
        return events
        
    def step(self, code: str):
        events = self.env.step(
            code,
            programs=self.program_context,
        )

        self.event_logger.step(events, code)
        return self._get_obs(events), 0, self.player_info['inventory'], {}
    
    def close(self):
        self.env.close()

    def _get_obs(self, events):
        print("#" * 50)
        print(events)
        self._update_chest_memory(events)
        self._update_player_info(events)
        return self.player_info
        
    def _update_chest_memory(self, events):
        chests = events[-1][1]["nearbyChests"]
        for position, chest in chests.items():
            if position in self.chest_memory:
                if isinstance(chest, dict):
                    self.chest_memory[position] = chest
                if chest == "Invalid":
                    print(
                        f"Action Agent removing chest {position}: {chest}"
                    )
                    self.chest_memory.pop(position)
            else:
                if chest != "Invalid":
                    print(f"Action Agent saving chest {position}: {chest}")
                    self.chest_memory[position] = chest

    def _update_player_info(self, events):
        self.player_info = events[-1][1]

    def tp_to_random_location(self):
        events = self.env.step(
            "await teleportToRandomGroundLocation(bot);",
            programs=self.program_context,
        )

    def clear_inventory(self):
        events = self.env.step(
            "await clearInventory(bot);",
            programs=self.program_context,
        )
